/*
 * Decompiled with CFR 0.152.
 */
package org.exist.storage.journal;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.text.DateFormat;
import org.apache.log4j.Logger;
import org.exist.EXistException;
import org.exist.storage.BrokerPool;
import org.exist.storage.journal.FileSyncThread;
import org.exist.storage.journal.LogException;
import org.exist.storage.journal.Loggable;
import org.exist.storage.journal.Lsn;
import org.exist.storage.lock.FileLock;
import org.exist.storage.txn.Checkpoint;
import org.exist.storage.txn.TransactionException;
import org.exist.util.ReadOnlyException;
import org.exist.util.sanity.SanityCheck;

public class Journal {
    private static final Logger LOG = Logger.getLogger((Class)Journal.class);
    public static final String RECOVERY_SYNC_ON_COMMIT_ATTRIBUTE = "sync-on-commit";
    public static final String RECOVERY_JOURNAL_DIR_ATTRIBUTE = "journal-dir";
    public static final String RECOVERY_SIZE_LIMIT_ATTRIBUTE = "size";
    public static final String PROPERTY_RECOVERY_SIZE_LIMIT = "db-connection.recovery.size-limit";
    public static final String PROPERTY_RECOVERY_JOURNAL_DIR = "db-connection.recovery.journal-dir";
    public static final String PROPERTY_RECOVERY_SYNC_ON_COMMIT = "db-connection.recovery.sync-on-commit";
    public static final String LOG_FILE_SUFFIX = "log";
    public static final String BAK_FILE_SUFFIX = ".bak";
    public static final String LCK_FILE = "journal.lck";
    public static final int LOG_ENTRY_HEADER_LEN = 11;
    public static final int LOG_ENTRY_BASE_LEN = 13;
    public static final int DEFAULT_MAX_SIZE = 0xA00000;
    private static final long MIN_REPLACE = 0x100000L;
    private int journalSizeLimit = 0xA00000;
    private FileChannel channel;
    private FileSyncThread syncThread;
    private Object latch = new Object();
    private File dir;
    private FileLock fileLock;
    private int currentFile = 0;
    private int inFilePos = 0;
    private ByteBuffer currentBuffer;
    private long currentLsn = -1L;
    private long lastLsnWritten = -1L;
    private long lastSyncLsn = -1L;
    private boolean inRecovery = false;
    private BrokerPool pool;
    private boolean syncOnCommit = true;

    public Journal(BrokerPool pool, File directory) throws EXistException {
        Integer sizeOpt;
        String logDir;
        this.dir = directory;
        this.pool = pool;
        this.currentBuffer = ByteBuffer.allocateDirect(0x100000);
        this.syncThread = new FileSyncThread(this.latch);
        this.syncThread.start();
        Boolean syncOpt = (Boolean)pool.getConfiguration().getProperty(PROPERTY_RECOVERY_SYNC_ON_COMMIT);
        if (syncOpt != null) {
            this.syncOnCommit = syncOpt;
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("SyncOnCommit = " + this.syncOnCommit));
            }
        }
        if ((logDir = (String)pool.getConfiguration().getProperty(PROPERTY_RECOVERY_JOURNAL_DIR)) != null) {
            File f = new File(logDir);
            if (!f.isAbsolute()) {
                if (pool.getConfiguration().getExistHome() == null) {
                    f = new File(pool.getConfiguration().getExistHome(), logDir);
                } else if (pool.getConfiguration().getConfigFilePath() != null) {
                    File confFile = new File(pool.getConfiguration().getConfigFilePath());
                    f = new File(confFile.getParent(), logDir);
                }
            }
            if (!f.exists()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("Output directory for journal files does not exist. Creating " + f.getAbsolutePath()));
                }
                try {
                    f.mkdirs();
                }
                catch (SecurityException e) {
                    throw new EXistException("Failed to create output directory: " + f.getAbsolutePath());
                }
            }
            if (!f.canWrite()) {
                throw new EXistException("Cannot write to journal output directory: " + f.getAbsolutePath());
            }
            this.dir = f;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Using directory for the journal: " + this.dir.getAbsolutePath()));
        }
        if ((sizeOpt = (Integer)pool.getConfiguration().getProperty(PROPERTY_RECOVERY_SIZE_LIMIT)) != null) {
            this.journalSizeLimit = sizeOpt * 1024 * 1024;
        }
    }

    public void initialize() throws EXistException, ReadOnlyException {
        File lck = new File(this.dir, LCK_FILE);
        this.fileLock = new FileLock(this.pool, lck.getAbsolutePath());
        boolean locked = this.fileLock.tryLock();
        if (!locked) {
            String lastHeartbeat = DateFormat.getDateTimeInstance(2, 2).format(this.fileLock.getLastHeartbeat());
            throw new EXistException("The journal log directory seems to be locked by another eXist process. A lock file: " + lck.getAbsolutePath() + " is present in the " + "log directory. Last access to the lock file: " + lastHeartbeat);
        }
    }

    public synchronized void writeToLog(Loggable loggable) throws TransactionException {
        if (this.currentBuffer == null) {
            throw new TransactionException("Database is shut down.");
        }
        SanityCheck.ASSERT(!this.inRecovery, "Write to log during recovery. Should not happen!");
        int size = loggable.getLogSize();
        int required = size + 13;
        if (required > this.currentBuffer.remaining()) {
            this.flushToLog(false);
        }
        this.currentLsn = Lsn.create(this.currentFile, this.inFilePos + this.currentBuffer.position() + 1);
        loggable.setLsn(this.currentLsn);
        try {
            this.currentBuffer.put(loggable.getLogType());
            this.currentBuffer.putLong(loggable.getTransactionId());
            this.currentBuffer.putShort((short)loggable.getLogSize());
            loggable.write(this.currentBuffer);
            this.currentBuffer.putShort((short)(size + 11));
        }
        catch (BufferOverflowException e) {
            throw new TransactionException("Buffer overflow while writing log record: " + loggable.dump(), e);
        }
    }

    public long lastWrittenLsn() {
        return this.lastLsnWritten;
    }

    public void flushToLog(boolean fsync) {
        this.flushToLog(fsync, false);
    }

    public synchronized void flushToLog(boolean fsync, boolean forceSync) {
        if (this.inRecovery) {
            return;
        }
        this.flushBuffer();
        if (forceSync || fsync && this.syncOnCommit && this.currentLsn > this.lastSyncLsn) {
            this.syncThread.triggerSync();
            this.lastSyncLsn = this.currentLsn;
        }
        try {
            if (this.channel.size() >= (long)this.journalSizeLimit) {
                this.pool.triggerCheckpoint();
            }
        }
        catch (IOException e) {
            LOG.warn((Object)"Failed to trigger checkpoint!", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void flushBuffer() {
        if (this.currentBuffer == null) {
            return;
        }
        Object object = this.latch;
        synchronized (object) {
            try {
                if (this.currentBuffer.position() > 0) {
                    this.currentBuffer.flip();
                    int size = this.currentBuffer.remaining();
                    while (this.currentBuffer.hasRemaining()) {
                        this.channel.write(this.currentBuffer);
                    }
                    this.currentBuffer.clear();
                    this.inFilePos += size;
                    this.lastLsnWritten = this.currentLsn;
                }
            }
            catch (IOException e) {
                LOG.warn((Object)"Flushing log file failed!", (Throwable)e);
            }
        }
    }

    public void checkpoint(long txnId, boolean switchLogFiles) throws TransactionException {
        block6: {
            LOG.debug((Object)"Checkpoint reached");
            this.writeToLog(new Checkpoint(txnId));
            if (switchLogFiles) {
                this.flushBuffer();
            } else {
                this.flushToLog(true, true);
            }
            try {
                if (!switchLogFiles || this.channel.position() <= 0x100000L) break block6;
                File oldFile = this.getFile(this.currentFile);
                RemoveThread rt = new RemoveThread(this.channel, oldFile);
                try {
                    this.switchFiles();
                }
                catch (LogException e) {
                    LOG.warn((Object)("Failed to create new journal: " + e.getMessage()), (Throwable)e);
                }
                rt.start();
            }
            catch (IOException e) {
                LOG.warn((Object)"IOException while writing checkpoint", (Throwable)e);
            }
        }
    }

    public void setCurrentFileNum(int fileNum) {
        this.currentFile = fileNum;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void switchFiles() throws LogException {
        ++this.currentFile;
        String fname = Journal.getFileName(this.currentFile);
        File file = new File(this.dir, fname);
        if (file.exists()) {
            boolean renamed;
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Journal file " + file.getAbsolutePath() + " already exists. Copying it."));
            }
            if ((renamed = file.renameTo(new File(file.getAbsolutePath() + BAK_FILE_SUFFIX))) && LOG.isDebugEnabled()) {
                LOG.debug((Object)("Old file renamed to " + file.getAbsolutePath()));
            }
            file = new File(this.dir, fname);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Creating new journal: " + file.getAbsolutePath()));
        }
        Object object = this.latch;
        synchronized (object) {
            this.close();
            try {
                FileOutputStream os = new FileOutputStream(file, true);
                this.channel = os.getChannel();
                this.syncThread.setChannel(this.channel);
            }
            catch (FileNotFoundException e) {
                throw new LogException("Failed to open new journal: " + file.getAbsolutePath(), e);
            }
        }
        this.inFilePos = 0;
    }

    public void close() {
        if (this.channel != null) {
            try {
                this.channel.close();
            }
            catch (IOException e) {
                LOG.warn((Object)"Failed to close journal", (Throwable)e);
            }
        }
    }

    public static final int findLastFile(File[] files) {
        int max = -1;
        for (int i = 0; i < files.length; ++i) {
            int p = files[i].getName().indexOf(46);
            String baseName = files[i].getName().substring(0, p);
            int num = Integer.parseInt(baseName, 16);
            if (num <= max) continue;
            max = num;
        }
        return max;
    }

    public File[] getFiles() {
        File[] files = this.dir.listFiles(new FilenameFilter(){

            public boolean accept(File dir, String name) {
                return name.endsWith(Journal.LOG_FILE_SUFFIX) && !name.endsWith("_index.log");
            }
        });
        return files;
    }

    public File getFile(int fileNum) {
        return new File(this.dir, Journal.getFileName(fileNum));
    }

    public void shutdown(long txnId) {
        if (this.currentBuffer == null) {
            return;
        }
        if (!BrokerPool.FORCE_CORRUPTION) {
            try {
                this.writeToLog(new Checkpoint(txnId));
            }
            catch (TransactionException e) {
                LOG.error((Object)("An error occurred while closing the journal file: " + e.getMessage()), (Throwable)e);
            }
            this.flushBuffer();
        }
        this.fileLock.release();
        this.syncThread.shutdown();
        try {
            this.syncThread.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        this.currentBuffer = null;
    }

    public void setInRecovery(boolean value) {
        this.inRecovery = value;
    }

    private static String getFileName(int fileNum) {
        String hex = Integer.toHexString(fileNum);
        hex = "0000000000".substring(hex.length()) + hex;
        return hex + '.' + LOG_FILE_SUFFIX;
    }

    private static class RemoveThread
    extends Thread {
        FileChannel channel;
        File file;

        RemoveThread(FileChannel channel, File file) {
            super("RemoveJournalThread");
            this.channel = channel;
            this.file = file;
        }

        public void run() {
            try {
                this.channel.close();
            }
            catch (IOException e) {
                LOG.warn((Object)("Exception while closing journal file: " + e.getMessage()), (Throwable)e);
            }
            this.file.delete();
        }
    }
}

